# Copyright 2018 VMware, Inc.  All rights reserved. -- VMware Confidential

import os
import sys
import unittest

CISKVUTILS_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
sys.path.append(CISKVUTILS_DIR)
from cis_kv_utils import TagAssociation


class TagAssociationTest(unittest.TestCase):
    def test_valid(self):
        rows = [
            ("tag_association urn:vmomi:HostSystem:host-10 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:Datastore:datastore-21 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:ClusterComputeResource:domain-c7 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:VirtualMachine:vm-36 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:VirtualMachine:vm-36:4f64c32f-0c56-4701-8b80-93f857a3875a InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("deleted_tag_association urn:vmomi:VirtualMachine:vm-36 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", "")
        ]
        for row in rows:
            ta, msg = TagAssociation.get_association(row[0])
            self.assertIsNotNone(ta)
            self.assertEqual(msg, "")

    def test_invalid(self):
        rows = [
            ("", "Unrecognized row"),
            ("tag_association VirtualMachine:vm-36 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:4f64c32f", "Unrecognized tag format"),
            ("tag_association urn:vmomi:VirtualMachine:vm-36:4f64c32f-0c56-4701-8b80-93f857a3875a urn:vmomi:InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", "Unrecognized tag format")
        ]
        for row in rows:
            ta, msg = TagAssociation.get_association(row[0])
            self.assertIsNone(ta)
            self.assertEqual(msg, row[1])

    def test_is_vpxd_tag_association(self):
        valid_entities_map = {
            "host": set([10, 20, 30]),
            "vm": set([36, 46]),
            "datastore": set([21])
        }
        rows = [
            ("tag_association urn:vmomi:HostSystem:host-10 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:VirtualMachine:vm-36 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:VirtualMachine:vm-36:4f64c32f-0c56-4701-8b80-93f857a3875a InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
        ]
        for row in rows:
            ta, msg = TagAssociation.get_association(row[0])
            self.assertIsNotNone(ta)
            self.assertTrue(ta.is_vpxd_tag_assn(valid_entities_map))

    def test_is_not_vpxd_tag_association(self):
        valid_entities_map = {
            "host": set([10, 20, 30]),
            "vm": set([36, 46])
        }
        rows = [
            ("tag_association urn:vmomi:Datastore:datastore-21 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("tag_association urn:vmomi:ClusterComputeResource:domain-c7 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", ""),
            ("deleted_tag_association urn:vmomi:VirtualMachine:vm-36 InventoryServiceTag:4f64c32f-0c56-4701-8b80-93f857a3875a:GLOBAL", "")
        ]
        for row in rows:
            ta, msg = TagAssociation.get_association(row[0])
            self.assertIsNotNone(ta)
            self.assertFalse(ta.is_vpxd_tag_assn(valid_entities_map))


if __name__ == '__main__':
    unittest.main()
