# Copyright 2022 VMware, Inc.
# All rights reserved. -- VMware Confidential

import os
import sys
import re
import logging

vmware_python_path = os.getenv('VMWARE_PYTHON_PATH')
if vmware_python_path and os.path.exists(vmware_python_path):
    sys.path.append(vmware_python_path)

from cis.utils import run_command


VMIDENTITY_RHTTP_CONF_FILE =\
    '/etc/vmware-rhttpproxy/endpoints.conf.d/vmidentity.conf'

logger = logging.getLogger(__name__)


def do_patching(ctx):

    # Only attempt the patching if the vmidentity.conf rhttpproxy file exists
    if not os.path.isfile(VMIDENTITY_RHTTP_CONF_FILE):
        logger.info('Config file for vmidentity rhttpproxy routing '
                    'does not exist: %s. Skipping patching ...' %
                    VMIDENTITY_RHTTP_CONF_FILE)
        return

    # Get the current rhttpproxy config file contents
    with open(VMIDENTITY_RHTTP_CONF_FILE, 'r') as file:
        conf_lines = file.readlines()

    # Look for the SSOCAC entry and change the HTTP access mode from 'request'
    # to 'allow' to prevent rhttpproxy from intercepting these requests
    updated_config = [re.sub(r'(^/websso/SAML2/SSOCAC\s.*)request$', r'\1allow', line)
                      for line in conf_lines]
    logger.info('Writing updated rhttpproxy config for vmidentity:\n%s' %
                updated_config)

    # Write the full updated config back to the same file
    with open(VMIDENTITY_RHTTP_CONF_FILE, 'w') as file:
        file.writelines(updated_config)

    # Trigger rhttpproxy to reload its configuration and pick up the new route,
    # if it is running
    (rc, stdout, _) = run_command(['pidof', 'rhttpproxy'])
    if rc == 0 and stdout:
        rhttpproxy_pid = stdout.strip()
        (rc, stdout, stderr) = run_command(['kill', '-1', rhttpproxy_pid])
        if rc != 0:
            _raise_error('Failed to kill rhttpproxy process with pid %s '
                         'to pick up new vmidentity SSOCAC routing' %
                         rhttpproxy_pid, stdout, stderr)


def _raise_error(msg, stdout=None, stderr=None):
    logger.error(msg)
    if stdout:
        logger.error(stdout)
    if stderr:
        logger.error(stderr)
    raise RuntimeError(msg)
