/* *************************************************************************
 * Copyright 2020-2021 VMware, Inc.   All rights reserved. -- VMware Confidential
 * *************************************************************************/

/*====================================================================================*/
/* VCDB_EXPAND.sql                                                                    */
/*====================================================================================*/
/*                                                                                    */
/* Description:                                                                       */
/*                                                                                    */
/* The purpose of this file is to be used in the EXPAND phase of the                  */
/* Non-Disruptive Upgrade process handling all DDL operations performed               */
/* in the VC schema during the upgrade from source to target. The                     */
/* operations added to this file only affect the final state the VC must              */
/* be in for the target to be operational after the upgrade has completed.            */
/*                                                                                    */
/* Instructions:                                                                      */
/*                                                                                    */
/* The way a change is going to be added to this file depends on its DDL              */
/* operation. To perform a change to VC schema for certain DDL operations             */
/* strict guidelines must be followed, in addition to know in which file              */
/* to add those changes same guideline must be reviewed. Please refer to              */
/* the link below to identify where your change must reside and how to                */
/* added it correctly.                                                                */
/*                                                                                    */
/* https://confluence.eng.vmware.com/display/VCUSOF/VC+Schema+Changes+Guide           */
/*                                                                                    */
/* Note!                                                                              */
/* Every CLN_ID added to this file MUST be also added to the tracking CLN_ID          */
/* table here:                                                                        */
/*                                                                                    */
/* https://confluence.eng.vmware.com/display/VCUSOF/NDU+-+Expand+-+CLN_ID+Track+Table */
/*                                                                                    */
/*====================================================================================*/

--Start of DB configurational settings

/* ON_ERROR_STOP- when set to true, the psql would stop on the first error and exit with status code 3*/
\set ON_ERROR_STOP on

/*============================================================================================*/
/*  Creates all NDU pre-defined procedure                                                     */
/*============================================================================================*/
BEGIN;
  \ir vcdb_ndu_procedures.sql
COMMIT;

/*============================================================================================*/
/*  GRANT permissions to proc library to other schemas                                        */
/*============================================================================================*/

CALL vcdb_ndu_db_schema_permissions();

--End of DB configurational settings

/* Begin 1 */
/*============================================================================================*/
/*  Add table VPX_HOST_SEVINFO and add column SEV_ENABLED to table VPX_NON_ORM_VM_CONFIG_INFO */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (1, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(1,'unknown',700,'
CREATE TABLE IF NOT EXISTS vc.VPX_HOST_SEVINFO (
   HOST_ID          BIGINT        not null,
   SEV_STATE        VARCHAR(32)   not null,
   SEV_MAX_GUESTS   INTEGER       not null,
   constraint PK_VPX_HOST_SEVINFO primary key (HOST_ID),
   constraint FK_VPX_HOST_SEVINFO foreign key (HOST_ID)
      references VPX_HOST (ID) on delete cascade)');$$,
$$CALL vcdb_ndu_add_column(1,'unknown',700,'vc', 'VPX_NON_ORM_VM_CONFIG_INFO', 'SEV_ENABLED', 'numeric(1)', 'default 0');$$
);
/*  End 1  */

/* Begin 2 */
/*============================================================================================*/
/*  Add columns CONTAINER_ID and ALIAS_OF to table VPX_DS_INFO                                */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (2, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(2,'unknown',700,'vc','VPX_DS_INFO', 'CONTAINER_ID', 'VARCHAR(33)');$$,
$$CALL vcdb_ndu_add_column(2,'unknown',700,'vc','VPX_DS_INFO', 'ALIAS_OF', 'VARCHAR(33)');$$
);
/*  End 2  */

/* Begin 3 */
/*============================================================================================*/
/*  Add columns DISK_PATH and USED_CAPACITY to table VPX_STORAGE_OBJECT_INFO                  */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (3, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(3,'vsan',700,'vc','VPX_STORAGE_OBJECT_INFO', 'DISK_PATH', 'TEXT');$$,
$$CALL vcdb_ndu_add_column(3,'vsan',700,'vc','VPX_STORAGE_OBJECT_INFO', 'USED_CAPACITY', 'BIGINT DEFAULT -1');$$
);
/*  End 3  */

/* Begin 4 */
/*============================================================================================*/
/*  Add column NUMA_ID to table VPX_HOST_PCI_DEVICE                                           */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (4, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(4,'unknown',700,'vc','VPX_HOST_PCI_DEVICE', 'NUMA_ID', 'INT');$$
);
/*  End 4  */

/* Begin 5 */
/*============================================================================================*/
/*  Add column LAG_TIMEOUT_MODE to table VPX_DVS_LAG                                          */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (5, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(5,'unknown',700,'vc','VPX_DVS_LAG', 'LAG_TIMEOUT_MODE', 'VARCHAR(255)');$$
);
/*  End 5  */

/* Begin 6 */
/*============================================================================================*/
/*  Add column VCLOCK to table VPX_FCD_DATASTORE_INFO                                         */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (6, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(6,'vsan',700,'vc','VPX_FCD_DATASTORE_INFO', 'VCLOCK', 'BIGINT DEFAULT -1');$$
);
/*  End 6  */

/* Begin 7 */
/*============================================================================================*/
/*  Create index VPX_STAT_COUNTER_D1 on column DEVICE_ID table VPX_STAT_COUNTER               */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (7, 700, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_crt_idx(7,'stats',700, 'vc', 'VPX_STAT_COUNTER_D1', 'VPX_STAT_COUNTER', '(DEVICE_ID)');$$
);
/*  End 7  */

/* Begin 8 */
/*============================================================================================*/
/*  Add table VPX_ATTESTD_SIGNING_INFO                                                        */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (8, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(8,'unknown',701,'
CREATE TABLE IF NOT EXISTS vc.VPX_ATTESTD_SIGNING_INFO (
   SERVICE_ID  BIGINT NOT NULL,
   CERT        TEXT,
   PUBKEY      TEXT,
   CONSTRAINT PK_VPX_ATTESTD_SIGNING_INFO PRIMARY KEY (SERVICE_ID),
   CONSTRAINT FK_VPX_ATTESTD_SIGNING_INFO FOREIGN KEY (SERVICE_ID)
      REFERENCES VPX_ATTESTD_SERVICES(ID) ON DELETE CASCADE)');$$
);
/*  End 8  */

/* Begin 9 */
/*============================================================================================*/
/*  Add table VPX_HOST_VSAN_RUNTIME                                                           */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (9, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(9,'unknown',701,'
CREATE TABLE IF NOT EXISTS vc.VPX_HOST_VSAN_RUNTIME (
   ID                   BIGINT  NOT NULL,
   VSAN_RUNTIME_INFO    TEXT,
   CONSTRAINT PK_VPX_HOST_VSAN_RUNTIME PRIMARY KEY (ID),
   CONSTRAINT FK_VPX_HOST_VSAN_RUNTIME_REF_VPX_HOST FOREIGN KEY (ID)
      REFERENCES VPX_HOST (ID) ON DELETE CASCADE)');$$
);
/*  End 9  */

/* Begin 10 */
/*=====================================================================================================*/
/*  Add columns VNUMA_CORES_PER_NODE and VNUMA_AUTO_CORES_PER_NODE to table VPX_NON_ORM_VM_CONFIG_INFO */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (10, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(10,'unknown',701,'vc','VPX_NON_ORM_VM_CONFIG_INFO', 'VNUMA_CORES_PER_NODE', 'INTEGER default 0');$$,
$$CALL vcdb_ndu_add_column(10,'unknown',701,'vc','VPX_NON_ORM_VM_CONFIG_INFO', 'VNUMA_AUTO_CORES_PER_NODE', 'NUMERIC(1) default 0');$$
);
/*  End 10  */

/* Begin 11 */
/*====================================================================================================*/
/*  Add table VPX_CRYPTO_NATIVE_KEY_PROVIDER and column BACKUP_COUNT to table VPX_CRYPTO_KMIP_CLUSTER */
/*====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (11, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(11,'unknown',701,'vc','VPX_CRYPTO_KMIP_CLUSTER', 'BACKUP_COUNT', 'INTEGER DEFAULT 0 NOT NULL');$$,
$$CALL vcdb_ndu_add_table(11,'unknown',701,'
CREATE TABLE IF NOT EXISTS vc.VPX_CRYPTO_NATIVE_KEY_PROVIDER (
   CLUSTER_ID     VARCHAR(255) NOT NULL,
   KEY_ID         VARCHAR(255) NOT NULL,
   ENC_KDK        VARCHAR(255) NOT NULL,
   KEY_ID_LAST_UPDATE TIMESTAMP,
   PERSIST_TOKEN  TEXT,
   CONSTRAINT PK_VPX_CRYPTO_NATIVE_KEY_PROVIDER PRIMARY KEY (CLUSTER_ID),
   CONSTRAINT FK_VPX_CRYPTO_NATIVE_KEY_PROVIDER FOREIGN KEY (CLUSTER_ID)
      REFERENCES VPX_CRYPTO_KMIP_CLUSTER (CLUSTER_ID) ON DELETE CASCADE)');$$
);
/*  End 11  */

/* Begin 12 */
/*=====================================================================================================*/
/*  This change is being executed in the vcdb_db_upgrade.sql(ID 12)                                    */
/*=====================================================================================================*/
/* End 12 */

/* Begin 13 */
/*=====================================================================================================*/
/*  Add column HA_ATTRIBUTES to table VPX_HOST_X                                                       */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (13, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(13,'unknown',701,'vc','VPX_HOST_X', 'HA_ATTRIBUTES', 'INTEGER NULL');$$
);
/*  End 13  */

/* Begin 14 */
/*=====================================================================================================*/
/*  Add column HA_ATTRIBUTES to table VPX_HOST_X                                                       */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (14, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(14,'unknown',701,'vc','VPX_CRYPTO_KMIP_CLUSTER', 'TPM_REQUIRED', 'INTEGER DEFAULT 0 NOT NULL');$$,
$$CALL vcdb_ndu_add_column(14,'unknown',701,'vc','VPX_CRYPTO_KMIP_CLUSTER', 'AUTO_REMEDIATE', 'INTEGER DEFAULT 0 NOT NULL');$$
);
/*  End 14  */

/* Begin 15 */
/*=====================================================================================*/
/* Migrate VPX_HOST.MASTER_GEN and VPX_HOST.MASTER_SPEC_GEN to VPX_HOST_SYNC_GEN table */
/*=====================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (15, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_mig_b2b_chg_65167();$$
);
/*  End 15  */

/* Begin 16 */
/*=====================================================================================================*/
/*  Add column CHIPSET_MOTHERBOARDLAYOUT to table VPX_NON_ORM_VM_CONFIG_INFO                           */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (16, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(16,'unknown',701,'vc','VPX_NON_ORM_VM_CONFIG_INFO', 'CHIPSET_MOTHERBOARDLAYOUT', 'VARCHAR(255) null');$$
);
/*  End 16  */

/* Begin 17 */
/*=====================================================================================================*/
/*  Add column APPHEARTBEAT_STATUS to table VPX_VM                                                     */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (17, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(17,'unknown',701,'vc','VPX_VM', 'APPHEARTBEAT_STATUS', 'VARCHAR(64) null');$$
);
/*  End 17  */

/* Begin 18 */
/*============================================================================================*/
/*  Add table VPX_HOST_SERVICE_MODE_STATE                                                     */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (18, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(18,'unknown',701,'
CREATE TABLE IF NOT EXISTS vc.VPX_HOST_SERVICE_MODE_STATE (
   HOST_ID              BIGINT  NOT NULL,
   SERVICE_MODE_STATES  TEXT NOT NULL,
   CONSTRAINT PK_VPX_HOST_SERVICE_MODE_STATE PRIMARY KEY (HOST_ID),
   CONSTRAINT FK_VPX_HOST_SERVICE_MODE_STATE FOREIGN KEY (HOST_ID)
      REFERENCES VPX_HOST (ID) ON DELETE CASCADE)');$$
);
/*  End 18  */

/* Begin 19 */
/*============================================================================================*/
/*  Add table VPX_VCS_VM                                                                      */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (19, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(19,'unknown',701,'
CREATE TABLE IF NOT EXISTS vc.VPX_VCS_VM (
   VM_ID              BIGINT  NOT NULL,
   VM_VCS_STATE       NUMERIC(1) DEFAULT 1   NOT NULL,
   CONSTRAINT PK_VPX_VCS_VM PRIMARY KEY (VM_ID),
   CONSTRAINT FK_VPX_VCS_VM FOREIGN KEY (VM_ID)
      REFERENCES VPX_VM (ID) ON DELETE CASCADE)');$$
);
/*  End 19  */

/* Begin 20 */
/*===========================================================================================*/
/* Migrate VPX_HOST.LAST_PERF_TIME and VPX_HOST.PERF_REFRESH_RATE to VPX_HOST_PERFINFO table */
/*===========================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (20, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_mig_b2b_chg_65173();$$
);
/*  End 20  */

/* Begin 21 */
/*=====================================================================================================*/
/*  Add column MAXIMUM_HW_VERSION to table VPX_DATACENTER and VPX_COMPUTE_RESOURCE                     */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (21, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(21,'unknown',701,'vc','VPX_DATACENTER', 'MAXIMUM_HW_VERSION', 'VARCHAR(255) null');$$,
$$CALL vcdb_ndu_add_column(21,'unknown',701,'vc','VPX_COMPUTE_RESOURCE', 'MAXIMUM_HW_VERSION', 'VARCHAR(255) null');$$
);
/*  End 21  */

/* Begin 22 */
/*=====================================================================================================*/
/*  Add column ENABLE_PMEM_FAILOVER to table VPX_VM                                                    */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (22, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(22,'unknown',701,'vc','VPX_VM', 'ENABLE_PMEM_FAILOVER', 'INT null');$$
);
/*  End 22  */

/* Begin 23 */
/*=======================================================================================================================================*/
/*  Add columns SNAPSHOT_DESCRIPTION, SNAPSHOT_ID and SNAPSHOT_DISK_PATH to table VPX_FCD_SNAPSHOT_INFO and create index VPX_SNAPSHOT_ID */
/*=======================================================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (23, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(23,'unknown',701,'vc','VPX_FCD_SNAPSHOT_INFO', 'SNAPSHOT_DESCRIPTION', 'VARCHAR(255)');$$,
$$CALL vcdb_ndu_add_column(23,'unknown',701,'vc','VPX_FCD_SNAPSHOT_INFO', 'SNAPSHOT_ID', 'VARCHAR(60)');$$,
$$CALL vcdb_ndu_add_column(23,'unknown',701,'vc','VPX_FCD_SNAPSHOT_INFO', 'SNAPSHOT_DISK_PATH', 'VARCHAR(255)');$$,
$$CALL vcdb_ndu_crt_idx(23,'vsan',701, 'vc','VPX_SNAPSHOT_ID', 'VPX_FCD_SNAPSHOT_INFO', '(SNAPSHOT_ID)', 'UNIQUE');$$
);
/*  End 23  */

/* Begin 24 */
/*============================================================================================*/
/*  Add table VPX_VCS_VM                                                                      */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (24, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(24,'unknown',701,'
CREATE TABLE IF NOT EXISTS vc.VPX_HOST_NFC_CONFIG (
   HOST_ID           BIGINT       NOT NULL,
   TOTAL_MEMORY      INTEGER      NOT NULL,
   STREAMING_MEMORY  INTEGER      NOT NULL,
   CONSTRAINT PK_VPX_HOST_NFC_CONFIG PRIMARY KEY (HOST_ID),
   CONSTRAINT FK_VPX_HOST_NFC_CONFIG FOREIGN KEY (HOST_ID)
      REFERENCES VPX_HOST (ID) ON DELETE CASCADE)');$$
);
/*  End 24  */

/* Begin 25 */
/*=======================================================================================================================================*/
/*  Add columns SNAPSHOT_DESCRIPTION, SNAPSHOT_ID and SNAPSHOT_DISK_PATH to table VPX_FCD_SNAPSHOT_INFO and create index VPX_SNAPSHOT_ID */
/*=======================================================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (25, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(25,'unknown',701,'vc','VPX_COMPUTE_RESOURCE', 'PMEM_ADMISSION_CTRL_ENABLED', 'BOOLEAN');$$,
$$CALL vcdb_ndu_add_column(25,'unknown',701,'vc','VPX_COMPUTE_RESOURCE', 'FAILOVER_PMEM_RESOURCES', 'INTEGER');$$,
$$CALL vcdb_ndu_add_column(25,'unknown',701,'vc','VPX_COMPUTE_RESOURCE', 'FAILOVER_PMEM_RESOURCE_AUTO_COMPUTE', 'BOOLEAN');$$
);
/*  End 25  */

/* Begin 26 */
/*============================================================================================*/
/*  Add table VPX_HOST_PARTIAL_MM_STATUS                                                      */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (26, 701, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_rnm_b2b_chg_65181();$$
);
/*  End 26  */

/* Begin 27 */
/*============================================================================================*/
/*  Add multiple table for VSAN HEALTH and index UK_VSAN_HISTORICAL_HEALTH                    */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (27, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(27,'vsan',702,'
CREATE TABLE IF NOT EXISTS vc.VSAN_HISTORICAL_CLUSTER (
   CLUSTER_ID             SERIAL,
   CLUSTER_NAME           VARCHAR(256),
   constraint PK_VSAN_HISTORICAL_CLUSTER primary key (
     CLUSTER_ID) using index TABLESPACE hs4,
   constraint UK_VSAN_HISTORICAL_CLUSTER unique (
      CLUSTER_NAME) using index TABLESPACE hs4
) TABLESPACE hs4');$$,
$$CALL vcdb_ndu_add_table(27,'vsan',702,'
CREATE TABLE IF NOT EXISTS vc.VSAN_HISTORICAL_GROUP (
   GROUP_ID             SERIAL,
   GROUP_NAME           VARCHAR(64),
   constraint PK_VSAN_HISTORICAL_GROUP primary key (
     GROUP_ID) using index TABLESPACE hs4,
   constraint UK_VSAN_HISTORICAL_GROUP unique (
     GROUP_NAME) using index TABLESPACE hs4
) TABLESPACE hs4');$$,
$$CALL vcdb_ndu_add_table(27,'vsan',702,'
CREATE TABLE IF NOT EXISTS vc.VSAN_HISTORICAL_TEST (
   TEST_ID             SERIAL,
   TEST_NAME           VARCHAR(64),
   constraint PK_VSAN_HISTORICAL_TEST primary key (
     TEST_ID) using index TABLESPACE hs4,
   constraint UK_VSAN_HISTORICAL_TEST unique (
     TEST_NAME) using index TABLESPACE hs4
) TABLESPACE hs4');$$,
$$CALL vcdb_ndu_add_table(27,'vsan',702,'
CREATE TABLE IF NOT EXISTS vc.VSAN_HISTORICAL_HEALTH_CHECK (
   HEALTH_ID           INT   NOT NULL,
   HEALTH_DESC         VARCHAR(15),
   constraint PK_VSAN_HISTORICAL_HEALTH_CHECK primary key (
     HEALTH_ID) using index TABLESPACE hs4,
   constraint UK_VSAN_HISTORICAL_HEALTH_CHECK unique (
     HEALTH_DESC) using index TABLESPACE hs4
) TABLESPACE hs4');$$,
$$CALL vcdb_ndu_add_table(27,'vsan',702,'
CREATE TABLE IF NOT EXISTS vc.VSAN_HISTORICAL_HEALTH (
   CLUSTER_ID          INT,
   GROUP_ID            INT,
   TEST_ID             INT,
   CREATION_TIME       TIMESTAMP   NOT NULL,
   HEALTH_ID           INT         NOT NULL,
   DETAILS             TEXT,
   TAGS                TEXT,
   constraint FK_CLUSTER_VHH_CLUSTERID foreign key(CLUSTER_ID)
               REFERENCES VSAN_HISTORICAL_CLUSTER (CLUSTER_ID),
   constraint FK_GROUP_VHH_GROUPID foreign key(GROUP_ID)
               REFERENCES VSAN_HISTORICAL_GROUP (GROUP_ID),
   constraint FK_TEST_VHH_TESTID foreign key(TEST_ID)
               REFERENCES VSAN_HISTORICAL_TEST (TEST_ID),
   constraint FK_HEALTH_VHH_HEALTHID foreign key(HEALTH_ID)
               REFERENCES VSAN_HISTORICAL_HEALTH_CHECK (HEALTH_ID),
   constraint UK_VSAN_HISTORICAL_HEALTH unique (
      CLUSTER_ID, GROUP_ID, TEST_ID, CREATION_TIME) using index TABLESPACE hs4
) PARTITION BY RANGE (CREATION_TIME) TABLESPACE hs4');$$
);
ALTER TABLE vc.VSAN_HISTORICAL_CLUSTER OWNER TO vc;
ALTER TABLE vc.VSAN_HISTORICAL_TEST OWNER TO vc;
ALTER TABLE vc.vsan_historical_group OWNER TO vc;
ALTER TABLE vc.VSAN_HISTORICAL_HEALTH_CHECK OWNER TO vc;
ALTER TABLE vc.VSAN_HISTORICAL_HEALTH OWNER TO vc;
/*  End 27  */

/* Begin 28 */
/*===============================================================================================*/
/*  Rename column IS_RESTRICTED_USER to IS_ONLY_MANAGED_BY_SP_USER and Add table VPX_EXT_UPGRADE */
/*===============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (28, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_rename_col(28,'vc-core',702,'vc','VPX_EXT', 'IS_RESTRICTED_USER', 'IS_ONLY_MANAGED_BY_SP_USER');$$,
$$CALL vcdb_ndu_add_table(28,'unknown',702,'
CREATE TABLE IF NOT EXISTS vc.VPX_EXT_UPGRADE(
   STATUS  NUMERIC(1) DEFAULT 0 not null)');$$
);
/*  End 28  */

/* Begin 29 */
/*=====================================================================================================*/
/*  This change is being executed in the vcdb_db_upgrade.sql(ID 29)                                    */
/*=====================================================================================================*/
/* End 29 */

/* Begin 30 */
/*===============================================================================================*/
/* Add table VPX_HOST_MEMORY_TIER_INFO                                                           */
/*===============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (30, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(30,'unknown',702,'
CREATE TABLE IF NOT EXISTS vc.VPX_HOST_MEMORY_TIER_INFO (
   HOST_ID             BIGINT         NOT NULL,
   MEMORY_TIERING_TYPE VARCHAR(56)    NOT NULL,
   MEMORY_TIER_INFO    TEXT           NOT NULL,
   CONSTRAINT PK_VPX_HOST_MEMORY_TIER_INFO PRIMARY KEY (HOST_ID),
   CONSTRAINT FK_VPX_HOST_MEMORY_TIER_INFO FOREIGN KEY (HOST_ID)
      REFERENCES VPX_HOST (ID) ON DELETE CASCADE)');$$
);
/*  End 30  */

/* Begin 31 */
/*==================================================================*/
/*  Create custom aggregate for stats_rollupX_proc                  */
/*==================================================================*/
--SELECT vcdb_ndu_cln_processor(
--$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (30, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,);
--DO
--$$
--BEGIN
--  IF NOT EXISTS (SELECT 1
--                   FROM information_schema.data_type_privileges
--                  WHERE object_schema = 'vc'
--                   AND object_name = 'vc_stat_agg_state')
--  THEN
--     CREATE TYPE vc_stat_agg_state AS (val1 numeric, val2 numeric, type numeric);
--  END IF;
--END $$
--LANGUAGE plpgsql;
--
--CREATE OR REPLACE FUNCTION vc_stat_agg_transition(agg_state   vc_stat_agg_state,
--                                                  val         numeric,
--                                                  type        numeric,
--                                                  sample_time timestamp)
--  RETURNS vc_stat_agg_state
--  LANGUAGE plpgsql
--  IMMUTABLE
--AS $$
--BEGIN
--  IF agg_state.type <> type THEN
--    raise exception 'type must have a consistent value for each group';
--  END IF;
--  IF val IS NULL THEN
--    RETURN agg_state;
--  END IF;
--  IF type < 0 or type > 4 THEN
--    RETURN agg_state;
--  END IF;
--  agg_state.type = type;
--  CASE type
--    WHEN 0 THEN
--      agg_state.val2 := agg_state.val2 + 1;
--      agg_state.val1 := coalesce(agg_state.val1, 0) + val;
--    WHEN 1 THEN
--      IF agg_state.val1 IS NULL OR val > agg_state.val1 THEN agg_state.val1 := val; END IF;
--    WHEN 2 THEN
--      IF agg_state.val1 IS NULL OR val < agg_state.val1 then agg_state.val1 := val; END IF;
--    WHEN 3 THEN
--      IF sample_time IS NULL THEN
--        RETURN agg_state;
--      END IF;
--      DECLARE
--        etime numeric;
--      BEGIN
--        etime := extract(epoch FROM sample_time)::numeric;
--        IF agg_state.val1 IS NULL OR etime > agg_state.val2 THEN
--          agg_state.val1 := val;
--          agg_state.val2 := etime;
--        END IF;
--      END;
--    WHEN 4 then
--      agg_state.val1 := coalesce(agg_state.val1, 0) + val;
--  END case;
--  RETURN agg_state;
--END;
--$$;
--
--
--CREATE OR REPLACE FUNCTION vc_stat_agg_final(agg_state vc_stat_agg_state)
--  RETURNS numeric
--  LANGUAGE plpgsql
--  IMMUTABLE
--AS $$
--BEGIN
--  RETURN
--    case agg_state.type
--      WHEN 0 THEN case when agg_state.val2 >= 1 THEN agg_state.val1 / agg_state.val2 ELSE NULL::numeric END
--      WHEN 1 THEN agg_state.val1
--      WHEN 2 THEN agg_state.val1
--      WHEN 3 THEN agg_state.val1
--      WHEN 4 THEN agg_state.val1
--      ELSE NULL::numeric
--    END;
--END;
--$$;
--
--
--DO
--$$
--BEGIN
--  IF NOT EXISTS (SELECT 1
--                   FROM information_schema.routines
--                  WHERE routine_schema = 'vc'
--                    AND routine_name = 'vc_stat_agg')
--  THEN
--      CREATE AGGREGATE vc_stat_agg(val numeric, type numeric, sample_time timestamp)
--      (
--        sfunc = vc_stat_agg_transition,
--        stype = vc_stat_agg_state,
--        finalfunc = vc_stat_agg_final,
--        initcond = '(,0,)'
--      );
--  END IF;
--END $$ LANGUAGE plpgsql;
--
--DROP FUNCTION IF EXISTS stats_rollup1_proc();
--/
--
--DROP FUNCTION IF EXISTS stats_rollup2_proc();
--/
--
--DROP FUNCTION IF EXISTS stats_rollup3_proc();
--/

/*  End 31  */

/* Begin 32 */
/*===============================================================================================*/
/* Add table VPX_VCLS_DRS_WF_COUNTERS                                                            */
/*===============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (32, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(32,'unknown',702,'
CREATE TABLE IF NOT EXISTS vc.VPX_VCLS_DRS_WF_COUNTERS(
   EVENT_ID             BIGINT         NOT NULL,
   CLUSTER_ID           BIGINT         NOT NULL,
   INIT_PLACE_COUNT     INT,
   PLACE_MULTIVM_COUNT  INT,
   LB_COUNT             INT,
   HOST_EMM_COUNT       INT,
   CLUSTER_EMM_COUNT    INT,
   XDRS_REC_COUNT       INT,
   RPE_REC_COUNT        INT,
   FIX_VIO_COUNT        INT,
   QUERY_WHATIF_COUNT   INT,
   CONSTRAINT PK_VPX_VCLS_DRS_WF_COUNTERS PRIMARY KEY (EVENT_ID),
   CONSTRAINT FK_VPX_VCLS_DRS_WF_COUNTERS FOREIGN KEY (CLUSTER_ID)
      REFERENCES VPX_ENTITY (ID) ON DELETE CASCADE
)');$$
);
/*  End 32  */

/* Begin 33 */
/*===============================================================================================*/
/* CLN_ID 33 IS NOT APPLICABLE, NO CHANGE ADDED WITH THIS ID                                     */
/*===============================================================================================*/
/*  End 33  */

/* Begin 34 */
/*=====================================================================================================*/
/*  This change is being executed in the vcdb_db_upgrade.sql(ID 34)                                    */
/*=====================================================================================================*/
/* End 34 */

/* Begin 35 */
/*==================================================================*/
/* Add new column NETWORK_OFFLOAD_ALLOWED in table VPX_DVS          */
/*==================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (35, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(35,'unknown',702,'vc','vpx_dvs', 'network_offload_allowed', 'NUMERIC(1) DEFAULT 0 NOT NULL');$$
);
/*  End 35  */

/* Begin 36 */
/*===============================================================================================*/
/* Add table VPX_VM_OP_NOTIFICATION_CFG                                                          */
/*===============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (36, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(36,'unknown',702,'
CREATE TABLE IF NOT EXISTS vc.VPX_VM_OP_NOTIFICATION_CFG (
   CLUSTER_ID                    BIGINT   NOT NULL,
   NOTIFICATION_TIMEOUT          INTEGER,
   CONSTRAINT PK_VPX_VM_OP_NOTIFICATION_CFG PRIMARY KEY (CLUSTER_ID),
   CONSTRAINT FK_VPX_VM_OP_NOTIFICATION_CFG_REF_VPX_ENTITY FOREIGN KEY (CLUSTER_ID)
      REFERENCES VPX_ENTITY (ID)
         ON DELETE CASCADE
)');$$
);
/*  End 36  */

/* Begin 37 */
/*==================================================================*/
/* Add new column NETWORK_OFFLOADING_ENABLED in table VPX_DVHOST    */
/*==================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (37, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(37,'unknown',702,'vc','VPX_DVHOST', 'NETWORK_OFFLOADING_ENABLED', 'NUMERIC(1) DEFAULT 0');$$
);
/*  End 37  */

/* Begin 38 */
/*===============================================================================================*/
/* Add table VPX_HOST_STATE_ENCRYPTION_INFO                                                      */
/*===============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (38, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(38,'unknown',702,'
CREATE TABLE IF NOT EXISTS vc.VPX_HOST_STATE_ENCRYPTION_INFO(
   HOST_ID                       BIGINT         NOT NULL,
   PROTECTION_MODE               VARCHAR(8)     NOT NULL,
   REQUIRE_SECURE_BOOT           NUMERIC(1),
   REQUIRE_EXEC_INSTALLED_ONLY   NUMERIC(1),
   CONSTRAINT PK_VPX_HOST_STATE_ENCRYPTION_INFO PRIMARY KEY (HOST_ID),
   CONSTRAINT FK_VPX_HOST_STATE_ENCRYPTION_INFO FOREIGN KEY (HOST_ID)
      REFERENCES VPX_HOST (ID) ON DELETE CASCADE
)');$$
);
/*  End 38  */

/* Begin 39 */
/*==================================================================*/
/* Add new column PREV_VERSION in table VPX_DVS                     */
/*==================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (39, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(39,'unknown',702,'vc','VPX_DVS', 'PREV_VERSION', 'VARCHAR(255)');$$
);
/*  End 39  */

/* Begin 40 */
/*==================================================================*/
/* Add cpuid.numSMT related columns for VPX_NON_ORM_VM_CONFIG_INFO  */
/*==================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (40, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(40,'unknown',702,'vc','VPX_NON_ORM_VM_CONFIG_INFO', 'CPUID_NUMSMT', 'INTEGER DEFAULT 1 NOT NULL');$$
);
/*  End 40  */

/* Begin 41 */
/*==================================================================*/
/* Add CRYPTO_INTEGRITY_PROTECTION_TYPE column for VPX_NON_ORM_VM_CONFIG_INFO  */
/*==================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (41, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(41,'unknown',702,'vc','VPX_NON_ORM_VM_CONFIG_INFO', 'CRYPTO_INTEGRITY_PROTECTION_TYPE', 'INTEGER DEFAULT 0');$$,
$$CALL vcdb_ndu_add_column(41,'unknown',702,'vc','VPX_VDEVICE_FILE_BACKING', 'CRYPTO_INTEGRITY_PROTECTION_TYPE', 'INTEGER DEFAULT 0');$$
);
/*  End 41  */

/* Begin 42 */
/*=====================================================*/
/* Add column NETWORK_OFFLOAD_SPEC_ID to table VPX_DVS */
/*=====================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (42, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(42,'vsphere-networking',702,'vc','VPX_DVS', 'NETWORK_OFFLOAD_SPEC_ID', 'VARCHAR(255)');$$
);
/*  End 42  */

/* Begin 52 */
/*==============================================================*/
/* Table: VPX_CLUSTER_VCLS_DATASTORE                            */
/*==============================================================*/
SELECT VCDB_NDU_CLN_PROCESSOR(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (52, 702, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(52,'vmacore', 702, '
CREATE TABLE IF NOT EXISTS VC.VPX_CLUSTER_VCLS_DATASTORE (
   CLUSTER_ID       BIGINT                 NOT NULL,
   DATASTORE_ID     BIGINT                 NOT NULL,
   ALLOWMENT        NUMERIC(1)   DEFAULT 0 NOT NULL,
   CONSTRAINT PK_VPX_CLUS_VCLS_DS PRIMARY KEY (DATASTORE_ID, CLUSTER_ID),
   CONSTRAINT FK_VPX_CLUS_VCLS_DS_REF_VPX_COMP_RES FOREIGN KEY (CLUSTER_ID)
         REFERENCES VPX_COMPUTE_RESOURCE (ID)
         ON DELETE CASCADE,
   CONSTRAINT FK_VPX_CLUS_VCLS_DS_REF_VPX_DS FOREIGN KEY (DATASTORE_ID)
         REFERENCES VPX_DATASTORE (ID)
         ON DELETE CASCADE)')$$
);
/* End 52 */

/* Begin 56 */
/*=====================================================================================================*/
/*  Add column PMEM_SNAPSHOT_MODE to table VPX_VM                                                     */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (56, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(56,'pmem',703,'vc','VPX_VM', 'PMEM_SNAPSHOT_MODE', 'VARCHAR(30) null');$$
);
/*  End 56  */

/* Begin 57 */
/*==============================================================*/
/* Table: VPX_HOST_SGXINFO                                      */
/*==============================================================*/
SELECT VCDB_NDU_CLN_PROCESSOR(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (57, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(57,'esx-sof-security',703,'vc','VPX_HOST_SGXINFO','SGX_REGISTRATION_INFO','TEXT null');$$
);
/* End 57 */

/* Begin 58 */
/*==============================================================*/
/* Table: VPX_VM_SGXINFO                                      */
/*==============================================================*/
SELECT VCDB_NDU_CLN_PROCESSOR(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (58, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(58,'esx-sof-security',703,'vc','VPX_VM_SGXINFO','SGX_REQUIRE_ATTESTATION','BOOLEAN null');$$
);
/* End 57 */

/* Begin 59 */
/*=====================================================================================================*/
/*  Add column CONFIG_MANAGER_ENABLED to table VPX_COMPUTE_RESOURCE                                    */
/*=====================================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (59, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(59,'unknown',703,'vc','VPX_COMPUTE_RESOURCE', 'CONFIG_MANAGER_ENABLED', 'NUMERIC(1) DEFAULT 0 NOT NULL');$$
);
/*  End 59  */

/* Begin 60 */
/*============================================================================================*/
/*  Add multiple tables for AUTHZ data cache                                                   */
/*============================================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (60, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_table(60,'vc-sof-security',703,'
CREATE TABLE IF NOT EXISTS VC.VPX_AUTHZ_PRIVILEGES (
   PRIV_ID              INT            NOT NULL,
   PRIV_NAME            VARCHAR(255)   UNIQUE  NOT NULL,
   PRIV_GROUP           VARCHAR(255)   NOT NULL,
   PRIV_VERSION         BIGINT         NOT NULL,
   PRIV_ON_PARENT       BOOLEAN        NOT NULL,
   CONSTRAINT PK_VPX_AUTHZ_PRIVILEGES PRIMARY KEY (PRIV_ID))');$$,
$$CALL vcdb_ndu_add_table(60,'vc-sof-security',703,'
CREATE TABLE IF NOT EXISTS VC.VPX_AUTHZ_ROLES (
   ROLE_ID              BIGINT            NOT NULL,
   ROLE_NAME            VARCHAR(255)      NOT NULL,
   ROLE_VERSION         BIGINT            NOT NULL,
   CONSTRAINT PK_VPX_AUTHZ_ROLES PRIMARY KEY (ROLE_ID))');$$,
$$CALL vcdb_ndu_add_table(60,'vc-sof-security',703,'
CREATE TABLE IF NOT EXISTS VC.VPX_AUTHZ_ROLES_PRIVILEGES (
   ROLE_ID              BIGINT                NOT NULL,
   PRIV_ID              INT                   NOT NULL,
   CONSTRAINT PK_VPX_AUTHZ_ROLES_PRIVILEGES PRIMARY KEY (ROLE_ID, PRIV_ID),
   CONSTRAINT FK_VPX_AUTHZ_ROLES_PRIVILEGES_ROLE FOREIGN KEY (ROLE_ID)
       REFERENCES VPX_AUTHZ_ROLES (ROLE_ID),
   CONSTRAINT FK_VPX_AUTHZ_ROLES_PRIVILEGES_PRIV FOREIGN KEY (PRIV_ID)
       REFERENCES VPX_AUTHZ_PRIVILEGES (PRIV_ID))');$$,
$$CALL vcdb_ndu_add_table(60,'vc-sof-security',703,'
CREATE TABLE IF NOT EXISTS VC.VPX_AUTHZ_GLOBAL_PERMISSIONS (
   PRINCIPAL            VARCHAR(255)      NOT NULL,
   IS_GROUP             BOOLEAN           NOT NULL,
   VERSION              BIGINT            NOT NULL,
   ROLE_ID              BIGINT            NOT NULL,
   CONSTRAINT PK_VPX_AUTHZ_GLOBAL_PERMISSIONS PRIMARY KEY (PRINCIPAL, IS_GROUP),
   CONSTRAINT FK_VPX_AUTHZ_GLOBAL_PERMISSIONS FOREIGN KEY (ROLE_ID)
       REFERENCES VPX_AUTHZ_ROLES (ROLE_ID))');$$
);
/*  End 60  */

/* Begin 61 */
/*===============================================================================*/
/*  Add column BOOT_OPTION_NETWORK_PROTOCOL to table  VPX_NON_ORM_VM_CONFIG_INFO */
/*===============================================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (61, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(61, 'vpxd-vmprov', 703, 'vc', 'VPX_NON_ORM_VM_CONFIG_INFO',
    'BOOT_OPTION_NETWORK_PROTOCOL', 'VARCHAR(5)');$$
);
/*  End 61*/

/* Begin 62 */
/*=====================================================*/
/*  Add GUEST_HEARTBEAT_STATUS column to VPX_VM table  */
/*=====================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (62, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(62,'vc-core',703,'vc','VPX_VM', 'GUEST_HEARTBEAT_STATUS', 'NUMERIC(1)', 'DEFAULT 0 NOT NULL CONSTRAINT vm_ghb_status_valid_check CHECK (GUEST_HEARTBEAT_STATUS IN (0, 1, 2, 3))');$$
);
/*  End 62  */

/* Begin 63 */
/*===================================================*/
/*  Add column INTEGRITY_CHECK_STATE to table VPX_VM */
/*===================================================*/
SELECT vcdb_ndu_cln_processor(
$$INSERT INTO VPX_VCDB_NDU_EXP (CLN_ID, RELEASE, CLN_STATUS) VALUES (63, 703, '') ON CONFLICT (CLN_ID) DO NOTHING;$$,
$$CALL vcdb_ndu_add_column(63,'vmcrypt',703,'vc','VPX_VM', 'INTEGRITY_CHECK_STATE', 'NUMERIC(1) null');$$
);
/*  End 63  */

/*=========================================================================*/
/*                                                                         */
/*               DO NOT ADD CHANGES AFTER THIS COMMENT BOX                 */
/*                                                                         */
/*=========================================================================*/
-----------------------------------------------------------------------------
-- END OF FILE
-----------------------------------------------------------------------------
