#!/usr/bin/env python
#
# Copyright 2016-2018 VMware, Inc.  All rights reserved. -- VMware Confidential
#

import os
import unittest
import logging
from distutils.version import LooseVersion

# Absolute import so that one can run the unitests from the endpoint-b2b folder
# with PYTHONPATH=$(dirname `pwd`) python3.5 -m unittest
# import_module is used because of the dash in the module's name.
from importlib import import_module
module = import_module('endpoint-b2b.version_util')

logging.basicConfig(level=logging.INFO)

class VersionUtilTestCase(unittest.TestCase):
   """
   Unit tests for the 'version_util' module
   """

   def setUp(self):
      current_dir = os.path.dirname(os.path.realpath(__file__))
      module.CONFIG_DIR = current_dir
      module.VERSION_FILE = os.path.join(current_dir, 'version.txt')
      module.TEMP_VERSION_FILE = os.path.join(current_dir, 'temp_version.txt')

   def tearDown(self):
      # Remove the version files generated by the tests

      if os.path.exists(module.VERSION_FILE):
         os.remove(module.VERSION_FILE)

      if os.path.exists(module.TEMP_VERSION_FILE):
         os.remove(module.TEMP_VERSION_FILE)

   def test_initialize_deletes_temp_version_leftover(self):
      """
      Test whether initialize removes a temp_version.txt that was left from a
      previous b2b update.
      """

      # Create the leftover file
      with open(module.TEMP_VERSION_FILE, 'w') as f:
         f.write('4.3.2.1')

      # Invoke the method to test
      module.initialize()

      # Assert temp version leftover is no more
      self.assertFalse(os.path.exists(module.TEMP_VERSION_FILE))

   def test_initialize_copies_version_file_to_temp(self):
      """
      Test whether initialize copies the version.txt to temp_version.txt.
      """

      # Create the version.txt file
      with open(module.VERSION_FILE, 'w') as f:
         f.write('2.0.0.0')

      # Invoke the method to test
      module.initialize()

      # Verify the contents of the temp file
      with open(module.TEMP_VERSION_FILE) as f:
         version_str = f.readline()
         self.assertEqual(version_str, '2.0.0.0')


   def test_get_version(self):
      """
      Test getting the version from the temp version file.
      """

      # Create 'temp_version.txt' in the current folder that contains the version string
      with open(module.TEMP_VERSION_FILE, 'w') as f:
         f.write('1.2.3.4')

      # Invoke the method to test
      version = module.get_version()

      # Verify the result
      self.assertEqual(version, LooseVersion('1.2.3.4'))

   def test_get_version_no_temp_version(self):
      """
      Test whether get_version returns 1.0.0.0 when no temp version file is
      present.
      """

      # Invoke the method to test
      version = module.get_version()

      # Verify the result
      self.assertEqual(version, LooseVersion('1.0.0.0'))


   def test_remove_temp_version(self):
      # Create a temporary version file
      with open(module.TEMP_VERSION_FILE, 'w') as f:
         f.write('1.2.3.4')

      # Invoke the method to test
      module.remove_temp_version()

      # Assert temp version is no more
      self.assertFalse(os.path.exists(module.TEMP_VERSION_FILE))
