#!/usr/bin/env python
# Copyright (c) 2022-2024 Broadcom. All Rights Reserved.
# Broadcom Confidential. The term "Broadcom" refers to Broadcom Inc.
# and/or its subsidiaries.
import fileinput, os, re, sys
import logging

SMS_CONFIG = 'sms-config.xml'
logger = logging.getLogger(__name__)

def getPropertyDict(filePath):
   """
   Read the .properties file and return a dictionary of key/value pairs.
   """
   logger.info("Inside sps_utils")
   logger.info("Read the .properties file and return a dictionary")
   propertyDict = {}
   with open(filePath, 'U') as f:
      for line in f:
         line = line.strip()
         # Skip comments and blank lines
         if not (line.startswith("#") or len(line)==0):
            key, value = line.split("=", 1)
            propertyDict[key.strip()] = value.strip()
   return propertyDict

def findAndReplace(fileName, searchString, replaceString):
   """
   Searches for the regex searchString in fileName and replaces it with replaceString.
   """
   searchPat = re.compile(searchString)
   if os.path.isfile(fileName):
      for line in fileinput.input(fileName, inplace = True):
         if re.search(searchPat, line):
            line = re.sub(searchPat, replaceString, line)
         sys.stdout.write(line)
      fileinput.close()

SPS_HOME = '/usr/lib/vmware-vpx/sps/'
SPS_CONF_PATH = os.path.join(SPS_HOME, 'conf')
SPS_SERVER_CONFIG_PATH = os.path.join(SPS_HOME, 'server', 'config')
SPS_LIB_PATH = os.path.join(SPS_HOME, 'lib')
CM_LIB_PATH = '/usr/lib/vmware-cm/lib'
# Get the override java security path for the OS
VMWARE_JAVA_SEC_PROP_PATH = os.path.normpath(
    os.path.join(os.environ['VMWARE_CFG_DIR'],
                 'java', 'vmware-override-java.security'))
LOG4J2_PROP_PATH = os.path.join(SPS_CONF_PATH,'log4j2-upgrade.properties')
COMMON_JARS=['bc-fips-1.0.2.1.jar','bcpkix-fips-1.0.5.jar','commons-beanutils-1.9.4.jar',
             'commons-codec-1.6.jar','commons-collections-3.2.2.jar',
             'commons-configuration-1.6.jar',
             'commons-lang-2.6.jar','commons-logging-1.1.3.jar',
             'custom-rolling-file-appender-1.0.jar','httpclient-4.5.3.jar',
             'httpcore-4.4.6.jar','jna.jar','joda-time-1.6.2.jar',
             'platform.jar','log4j-api-2.21.1.jar','log4j-core-2.21.1.jar',
             'log4j-slf4j-impl-2.21.1.jar','log4j-jcl-2.21.1.jar','slf4j-api-1.7.30.jar',
             'spring-aop-5.3.30.jar','spring-beans-5.3.30.jar',
             'spring-context-5.3.30.jar','spring-core-5.3.30.jar',
             'spring-expression-5.3.30.jar','spring-oxm-5.3.30.jar',
             'spring-web-5.3.30.jar','spring-webmvc-5.3.30.jar',
             'opentracing-util-0.32.0.jar','opentracing-noop-0.32.0.jar',
             'opentracing-api-0.32.0.jar','jaeger-core-1.6.0.jar','jaeger-thrift-1.6.0.jar',
         'libthrift-0.14.1.jar']

# Pickup the Java path
def getJavaCommand():
  javaCommand = os.environ['VMWARE_JAVA_HOME'] + os.sep + "bin" + os.sep + "java"
  logger.info("Picked up Java command as per VMWARE_JAVA_HOME as %s", javaCommand)
  return javaCommand

# Add required common libraries from common jars directory
def addCommonLibs(commonJarsPath):
  # Start constructing command classpath from conf directories
  logger.info("Adding required common libraries from common jars directory")
  upgradeClassPath = SPS_CONF_PATH + os.pathsep + SPS_SERVER_CONFIG_PATH

  # Add server libraries from lib directory
  upgradeClassPath = upgradeClassPath + os.pathsep + SPS_LIB_PATH + os.sep + "*"

  # Add required common libraries from common jars directory
  for jarFile in COMMON_JARS:
    upgradeClassPath = upgradeClassPath + os.pathsep + commonJarsPath + os.sep + jarFile
  return upgradeClassPath
