#########################################################################
#
#   Copyright (c) 2023-2024 Broadcom. All Rights Reserved.
#   Broadcom Confidential. The term "Broadcom" refers to Broadcom Inc.
#   and/or its subsidiaries.
#
#########################################################################
from logging import getLogger
from os.path import join, isfile
from cis.utils import (read_properties, replace_properties_in_file,
                       write_props_to_file)
from .patchUtil import (TO_PREPARE_SUFFIX, updateVersion,
                        addNewLineBeforeFirstElement)


EAM_CFG_DIR = "/etc/vmware-eam/"
LOG4J_2_TOMCAT_PROPS = "log4j2-tomcat.properties"
LOG4J_2_TOMCAT_PROPS_OLD = join(EAM_CFG_DIR, LOG4J_2_TOMCAT_PROPS)
LOG4J_2_TOMCAT_PROPS_NEW = join(EAM_CFG_DIR, LOG4J_2_TOMCAT_PROPS + TO_PREPARE_SUFFIX)

CONFIG_TO_ADD = {
   "appender.catalina.strategy.type": "DefaultRolloverStrategy",
   "appender.catalina.strategy.max": "20",
   "appender.localhost.strategy.type": "DefaultRolloverStrategy",
   "appender.localhost.strategy.max": "20"
}

CONFIG_TO_REPLACE = {
   "appender.catalina.filePattern": "${tomcat.logdir}/catalina-${fileDatePattern}.gz",
   "appender.localhost.filePattern": "${tomcat.logdir}/localhost-${fileDatePattern}.gz"
}


_log = getLogger(__name__)
def run(ctx):
   """
   :param: ctx : B2B context
   :type: ctx : PatchContext, B2B specific structure, please refer to B2B
               documentation.
   :type: bool
   """
   _log.info("Updating log42-tomcat.properties log rotations.")
   props = {}
   if isfile(LOG4J_2_TOMCAT_PROPS_OLD):
      props = read_properties(LOG4J_2_TOMCAT_PROPS_OLD)
   else:
      props = read_properties(LOG4J_2_TOMCAT_PROPS_NEW)

   propsToAdd = {}
   propsToReplace = {}
   for key, value in CONFIG_TO_ADD.items():
      if key not in props.keys():
         propsToAdd.update({key: value})
   if propsToAdd:
      write_props_to_file(LOG4J_2_TOMCAT_PROPS_NEW,
                          addNewLineBeforeFirstElement(propsToAdd.items()),
                          'a+')

   for key, value in CONFIG_TO_REPLACE.items():
      if value not in props.values():
         propsToReplace.update({key: value})
   if propsToReplace:
      replace_properties_in_file(LOG4J_2_TOMCAT_PROPS_NEW, propsToReplace)

   # We do not update EAM version. so that patches from 8.x can be applied correctly.
   # Respective patch from 8.x is patch8 and will be applied correctly even if applied
   # previously here in 7.x line. We just update the files so we can get the rotation.


def updateReplicationConfig(replicationConfig):
   """
   Updates given replication config with patch specific config
   file replication configuration.

   Each item in the dictionary have following semantic:
      Key -> path of a file from source machine
      Value -> path of a file from target machine
   The file from source machine is replicated on the location pointed
   in the value over the target machine. If the Value is None, the file
   is not replicated.

   :param: replicationConfig: dictionary where to place patch specific
           config replication.
   :type: dict(str, str)
   """
   replicationConfig[LOG4J_2_TOMCAT_PROPS_OLD] = None
   replicationConfig[LOG4J_2_TOMCAT_PROPS_NEW] = LOG4J_2_TOMCAT_PROPS_OLD
