#########################################################################
#
#   Copyright 2019-2021,2023 VMware, Inc.  All rights reserved. VMware Confidential
#
#########################################################################
from os.path import join
from cis.utils import readprop, write_props_to_file
from logging import getLogger
from .patchUtil import (TO_PREPARE_SUFFIX, updateVersion,
                        addNewLineBeforeFirstElement)

VERSION = 1
EAM_CFG_DIR = "/etc/vmware-eam/"
EAM_PROPS = "eam-vim.properties"
EAM_VERSION = "version"
EAM_VIM_PROPS_FILE_OLD = join(EAM_CFG_DIR, EAM_PROPS)
EAM_VIM_PROPS_FILE_NEW = EAM_VIM_PROPS_FILE_OLD +\
                         TO_PREPARE_SUFFIX
EAM_VERSION_OLD = join(EAM_CFG_DIR, EAM_VERSION)
EAM_VERSION_NEW = join(EAM_CFG_DIR, EAM_VERSION + TO_PREPARE_SUFFIX)

LS_URL_KEY = "ls.url"
LS_NODE_ID_KEY = "ls.node.id"
EMPTY = ""

_log = getLogger(__name__)


def run(ctx, isExpand):
   """
   Adds LookupService property keys to the eam-vim.properties file.

   :param: ctx : B2B context
   :type: ctx : PatchContext, B2B specific structure, please refer to B2B
               documentation.
   :param: isExpand: boolean flag, True if this patch is invoked from expand
                    phase, False otherwise.
   :type: bool
   """

   file = EAM_VIM_PROPS_FILE_NEW if isExpand else EAM_VIM_PROPS_FILE_OLD
   versionFile = EAM_VERSION_NEW if isExpand else EAM_VERSION_OLD
   _log.info("Adding property keys ( 'ls.url=' and 'ls.node.id=' ) \
               for LookupService to the " + file)
   try:
      lsUrl = readprop(file, LS_URL_KEY)
      lsNode = readprop(file, LS_NODE_ID_KEY)
      if lsUrl or lsUrl == EMPTY:
         _log.info(LS_URL_KEY + " already added: " + lsUrl)
      else:
         prop = {LS_URL_KEY: EMPTY}
         write_props_to_file(EAM_VIM_PROPS_FILE_NEW,
                             addNewLineBeforeFirstElement(prop.items()),
                             mode="a+")

      if lsNode or lsNode == EMPTY:
         _log.info(LS_NODE_ID_KEY + " already added: " + lsNode)
      else:
         prop = {LS_NODE_ID_KEY: EMPTY}
         write_props_to_file(EAM_VIM_PROPS_FILE_NEW,
                             addNewLineBeforeFirstElement(prop.items()),
                             mode="a+")

      updateVersion(VERSION, versionFile)
      _log.info("The LookupService property keys were added successfully")
   except Exception as e:
      _log.error("EAM version one patch failed.", e)
      raise e

def updateReplicationConfig(replicationConfig):
   """
   Updates given replication config with patch specific config
   file replication configuration.

   Each item in the dictionary have following semantic:
      Key -> path of a file from source machine
      Value -> path of a file from target machine
   The file from source machine is replicated on the location pointed
   in the value over the target machine. If the Value is None, the file
   is not replicated.

   :param: replicationConfig: dictionary where to place patch specific
           config replication.
   :type: dict(str, str)
   """
   replicationConfig[EAM_VERSION_OLD] = None
   replicationConfig[EAM_VERSION_NEW] = EAM_VERSION_OLD
   replicationConfig[EAM_VIM_PROPS_FILE_OLD] = None
   replicationConfig[EAM_VIM_PROPS_FILE_NEW] = EAM_VIM_PROPS_FILE_OLD
