#!/usr/bin/env python

#
# Copyright 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
#
"""
Patch script that does the following:
Add gRPC endpoints for Certificate Management

"""
import logging
import os
import re
import sys
import warnings

vmware_python_path = os.getenv('VMWARE_PYTHON_PATH')
if vmware_python_path and os.path.exists(vmware_python_path):
    sys.path.append(vmware_python_path)

from cis.cisreglib import LookupServiceClient, SsoClient, VmafdClient, _get_syscfg_info

import cis.cisreglib as crlib
from fss_utils import getTargetFSS

logger = logging.getLogger(__name__)

# Suppress warnings coming from vmafd and identity imports.
with warnings.catch_warnings():
    warnings.simplefilter('ignore', RuntimeWarning)
    from identity.vmkeystore import VmKeyStore

CM_PRODUCT_ID = 'com.vmware.certificatemanagement'
CM_TYPE_ID = 'certificatemanagement'
MACHINE_NAME = 'machine'
# PATCH_VERSION = '3'
COMPONENT_NAME = 'certificatemanagement'
SYNC_PROPERTY_NAME = 'Syncable'
SYNC_PROPERTY_VALUE = 'ELM,SPOG'
GRPC_ENDPOINT_PROTOCOL = 'gRPC'
GRPC_ENDPOINT_PORT = 4002
SUBSCRIBABLE_ATTRIBUTE_NAME = 'Subscribable'
SUBSCRIBABLE_ATTRIBUTE_TRUE_VALUE = 'true'

def doPatching(ctx):
    if getTargetFSS("HybridVC_SyncaaS"):
        logger.info("CertificateManagement patch: being executed %s", ctx)
        logger.info("Updating grpc endpoint.")
        update_endpoints()

def get_config_dir():
    return '/usr/lib/vmware-%s/config' % COMPONENT_NAME


def update_endpoints():

    logger.info("Connecting to Lookup Service")
    ls_url, domain_name = _get_syscfg_info()
    ls_obj = LookupServiceClient(ls_url, retry_count=5)
    logger.info("Getting STS endpoint")
    sts_url, sts_cert_data = ls_obj.get_sts_endpoint_data()

    logger.info("Logging into SSO AdminClient as machine solution user")
    cert = None
    key = None

    try:
        with VmKeyStore('VKS') as ks:
            ks.load(MACHINE_NAME)
            cert = ks.get_certificate(MACHINE_NAME)
            key = ks.get_key(MACHINE_NAME)

        sso_client = SsoClient(sts_url, sts_cert_data, None, None, cert=cert,
                               key=key)
        ls_obj.set_sso_client(sso_client)

        logger.info("Fetching service Info for the CertificateManagement from Lookup Service")
        cert_mgmt_service_info = ls_obj.get_local_service_info(CM_PRODUCT_ID, CM_TYPE_ID)

        # Get the service info in the format the reregister API accepts
        service_info = ls_obj.service_content.serviceRegistration.Get(cert_mgmt_service_info.serviceId)
        mutable_spec = ls_obj._svcinfo_to_setspec(service_info)

        logger.info("Adding Endpoint and Syncable property to CertificateManagement")
        addSyncableAttribute(mutable_spec)
        logger.info("Adding Subscribable  property to CertificateManagement")
        addSubscribableAttribute(mutable_spec)
        addGrpcEndpoint(mutable_spec)
        ls_obj.reregister_service(service_info.serviceId, mutable_spec)

        sso_client.cleanup()
    except BaseException as e:
        logger.error("Failed to reregister CertificateManagement with Lookup Service")
        raise e

def addSyncableAttribute(mutable_spec):
    attribute = crlib.lookup.ServiceRegistration.Attribute()
    attribute.key = SYNC_PROPERTY_NAME
    attribute.value = SYNC_PROPERTY_VALUE
    mutable_spec.serviceAttributes.append(attribute)

def addSubscribableAttribute(mutable_spec):
    subscribablAttribute = crlib.lookup.ServiceRegistration.Attribute()
    subscribablAttribute.key = SUBSCRIBABLE_ATTRIBUTE_NAME
    subscribablAttribute.value = SUBSCRIBABLE_ATTRIBUTE_TRUE_VALUE
    mutable_spec.serviceAttributes.append(subscribablAttribute)

def addGrpcEndpoint(mutable_spec):
    endpoint = crlib.lookup.ServiceRegistration.Endpoint()

    vmafd_client = VmafdClient()
    endpoint_url = 'https://' + vmafd_client.get_pnid() + ':' \
                   + str(GRPC_ENDPOINT_PORT)
    logger.info("Endpoint Url for gRPC endpoint: " + endpoint_url)
    endpoint.url = endpoint_url

    endpoint_type = crlib.lookup.ServiceRegistration.EndpointType()
    endpoint_type.protocol = GRPC_ENDPOINT_PROTOCOL
    endpoint_type.type = CM_TYPE_ID
    endpoint.endpointType = endpoint_type
    endpoint.sslTrust = mutable_spec.serviceEndpoints[0].sslTrust

    mutable_spec.serviceEndpoints.append(endpoint)
