# Copyright 2012 Vmware, Inc. All rights reserved. -- VMware Confidential
#
'''
Base exception for upgrade and install.
'''
import json
from . import json_utils
from .l10n import localizedString
from .msgL10n import MessageMetadata as _T
from .componentStatus import ErrorInfo


def get_default_error():
   return localizedString(_T("install.default.failure.message",
                             'The installation of vCenter Server failed due to an internal error.'))

def get_default_resolution():
   return localizedString(_T("install.default.failure.resolution",
                             'This is an unrecoverable error, please retry install. If you encounter '
                             'this error again, please search for these symptoms in the '
                             'VMware Knowledge Base for any known issues and possible resolutions. '
                             'If none can be found, collect a support bundle and open a support request.'))

class BaseInstallException(Exception):
   '''Represents the base exception type of install/upgrade specific error.
   '''
   def __init__(self, errInfo):
      '''Initializes the error.

      @param errInfo: details describing the error condition and resolution.
      @type problemId: ErrorInfo
      '''
      Exception.__init__(self)
      self._info = errInfo

   def appendErrorStack(self, cause):
      '''Append information to error stack

      @param cause: error message
      @type: LocalizableMessage
      '''
      self._info.appendErrorDetail(cause)

   def getErrorInfo(self):
      '''Return the error information of the exception.
      '''
      return self._info

   def __str__(self):
      return json.dumps(self._info,
                        default=json_utils.ObjectJsonConverter(False).encode,
                        indent = 4)

class Timedout(BaseInstallException):
   def __init__(self, errStr):
      '''Initializes the error.
      @param errStr: describing the error condition.
      @type errStr: string
      '''
      timeout_error = _T('install.framework.timedout',
                         "Execution timed out: '%s'")
      error_msg = localizedString(timeout_error, errStr)

      errInfo = ErrorInfo([error_msg])
      BaseInstallException.__init__(self, errInfo)
