#!/usr/bin/python
# Copyright 2014 VMware, Inc.  All rights reserved. -- VMware Confidential
#

import os
import logging
try:
    from status_reporting_sdk.msgL10n import L10nMessage, MessageMetadata, Catalog
    from status_reporting_sdk.json_utils import Serializable
except: #pylint: disable=W0702
    #Applmgmt uses this l10n and there there is no status_reporting_sdk module but it is in the same
    #package as the rest
    from .msgL10n import L10nMessage, MessageMetadata, Catalog #pylint: disable=E0402
    from .json_utils import Serializable #pylint: disable=E0402

# The list of supported locales is documented in the B2B functional spec
# The locale names matches the locale directory names under
# bora/install/cis_patch_locale

DEFAULT_LOCALE = 'en'

B2B_VMSG_FILE_NAME = 'cis_patch_catalog.vmsg'

logger = logging.getLogger(__name__)

_theCatalog = None

def configure(localeDir, locale=DEFAULT_LOCALE):
    # pylint: disable=W0613
    '''Configures the l10n module. The method initialzes the module level
    global catalog. It loads two files from the localeDir, the English vmsg
    file and the translation vmsg file identfiied by the locale parameter.

    @param localeDir: The directory where the localization catalog is located.
    @type localeDir: str

    @param locale: current locale
    @type locale: str
    '''
    enVmsgPath = os.path.join(localeDir, DEFAULT_LOCALE, B2B_VMSG_FILE_NAME)
    loVmsgPath = os.path.join(localeDir, locale, B2B_VMSG_FILE_NAME)
    global _theCatalog
    _theCatalog = Catalog(enVmsgPath, loVmsgPath)

class LocalizableMessage(Serializable):
    def __init__(self, l10nMsg):
        self.l10nMsg = l10nMsg

    def __str__(self):
        return self.l10nMsg.translate(_theCatalog)

    def __serialize__(self):
        return self.l10nMsg.jsonEncode(_theCatalog)

    def __localizableMessage__(self):
        return self.l10nMsg

    @classmethod
    def __deserialize__(cls, jObject):
        l10nMsg = L10nMessage.jsonDecode(jObject)
        return cls(l10nMsg)

    def __eq__(self, other):
        if not isinstance(other, LocalizableMessage):
            return False
        return self.l10nMsg.msgMeta.msgId ==\
            other.l10nMsg.msgMeta.msgId

    def __ne__(self, other):
        return not self.__eq__(other)

    def __hash__(self):
        return hash(self.l10nMsg.msgMeta.msgId)

def localizedString(msgMeta, args=None):
    '''Translates message based on the current locale.

    @param msgMeta: a message metadata definition that
                    includes an id and a format text. See msgL10n.py
    @type  msgKey: MessageMetadata

    @param args: Message arguments which need to be substituted.

    @return: A serializable and localizable message object
    @rtype: LocalizableMessage
    '''
    l10nMsg = L10nMessage(msgMeta, args)
    return LocalizableMessage(l10nMsg)

def msgMetadata(msgId, formatText):
    '''Wrapper function that contructs a MessageMetadata
    object. Please see the msgL10n module for the definition of
    the MessageMetadata class
    @param msgId: The message id is dot seperated string representation
                  such as framework.esx.incompatible.
    @param formatText: The format text is the standard python string format.
    @return: a MessageMetadata object
    @rtypoe: MessageMetadata
    '''
    return MessageMetadata(msgId, formatText)
